* racd10.do  January 2013 for Stata version 12

capture log close
log using racd10.txt, text replace

********** OVERVIEW OF racd10.do **********

* STATA Program 
* copyright C 2013 by A. Colin Cameron and Pravin K. Trivedi 
* used for "Regression Analyis of Count Data" SECOND EDITION
* by A. Colin Cameron and Pravin K. Trivedi (2013)
* Cambridge University Press

* This STATA program analyzes doctor visits data for chapter 10
*  10.5 EXAMPLE: DOCTOR VISITS AND HEALTH INSURANCE

* To run you need file
*   racd10data.dta
* in your directory

* And you need Stata user-written program
*   mtreatreg
* in your directory

* To speed up program reduce the number of bootstraps

********** SETUP **********

set more off
version 12
clear all
* set mem 10m
* set linesize 82
set scheme s1mono  /* Graphics scheme */
* set maxvar 100 width 1000
 
********** DATA DESCRIPTION

* The original data are from 
* P. Deb and P.K. Trivedi (2006a), "Specification and simulated 
* likelihood estimation of a nonnormal treatment-outcome model with
* selection: application to health care utilization," 
* Econometrics Journal 9: 307-331.
* See this article for more detailed discussion 

********** 10.5.1 READ DATA, DROP OBSERVATIONS, AND SUMMARIZE 

use racd10data.dta, clear

* Drop some observations before analysis
drop if docvis > 70
drop if private==1 & medicaid==1

describe
summarize
tabulate docvis

********** 10.5.2 ONE ENDOGENOUS REGRESSOR (private)

*** No endogeneity

global XLISTEXOG medicaid age age2 educyr actlim totchr

* Poisson
poisson docvis private $XLISTEXOG, vce(robust)
estimates store P

* Negative binomial
nbreg docvis private $XLISTEXOG, vce(robust)
estimates store NB2

*** Control function for Endogeneity

* First stage regression
regress private income ssiratio $XLISTEXOG, vce(robust)
predict pres, residuals
generate pressq = pres^2

* Poisson with residuals added
* Standard errors here do not control for 2-step estimation
* Instead use the bootstrap below 
poisson docvis private $XLISTEXOG pres, vce(robust)
estimates store PCF

* Negative binomial with residuals added 
* Standard errors here do not control for 2-step estimation
* Instead use the bootstrap below 
nbreg docvis private $XLISTEXOG pres, vce(robust)
estimates store NB2CF

* Program to return b for Poisson 2-step estimator of selection model 
program poisson2step, eclass
  version 12
  tempname b V
  capture drop pres pressq
  regress private income ssiratio $XLISTEXOG, vce(robust)
  predict pres, residuals
  generate pressq = pres^2
  poisson docvis private $XLISTEXOG pres
  matrix `b' = e(b)
  ereturn post `b' 
end
* Check preceding program by running once
poisson2step
ereturn display
* Bootstrap for Poisson two-step estimator
bootstrap _b, reps(400) seed(10101) nowarn: poisson2step
estimates store PCFboot

* Program to return b for Negative binomial 2-step estimator of selection model 
program NB2step, eclass
  version 12
  tempname b V
  capture drop pres pressq
  regress private income ssiratio $XLISTEXOG, vce(robust)
  predict pres, residuals
  generate pressq = pres^2
  nbreg docvis private $XLISTEXOG pres
  matrix `b' = e(b)
  ereturn post `b' 
end
* Check preceding program by running once
NB2step
ereturn display
* Bootstrap for Poisson two-step estimator
bootstrap _b, reps(400) seed(10101) nowarn: NB2step
estimates store NB2CFboot

*** GMM for endogeneity

* Additive heterogeneity - one step GMM
gmm (docvis - exp({xb:private $XLISTEXOG}+{b0})),              ///
  instruments(income ssiratio $XLISTEXOG) onestep vce(robust)
estimates store GMMadd1S

* Additive heterogeneity - two step GMM
gmm (docvis - exp({xb:private $XLISTEXOG}+{b0})),              ///
  instruments(income ssiratio $XLISTEXOG) twostep vce(robust)
estimates store GMMadd2S
estat overid

* Multiplicative heterogeneity - one step GMM 
gmm ( (docvis / exp({xb:private $XLISTEXOG}+{b0})) - 1),      ///
  instruments(income ssiratio $XLISTEXOG) onestep vce(robust)
estimates store GMMmult1S

* Multiplicative heterogeneity - two step GMM
gmm ( (docvis / exp({xb:private $XLISTEXOG}+{b0})) - 1),      ///
  instruments(income ssiratio $XLISTEXOG) twostep vce(robust)
estimates store GMMmult2S
estat overid

*** RESULTS: TABLE 10.1

*** TABLE 10.1 - Poisson and NB models
estimates table P NB2 PCFboot NB2CFboot, b(%7.4f) se(%7.3f) stats(N ll) ///
   stfmt(%9.1f) modelwidth(9) equations(1)

*** TABLE 10.1 (Continued) - GMM (two-step estimates used in Table 10.1)
estimates table GMMadd1S GMMmult1S GMMadd2S GMMmult2S, b(%7.4f) se(%7.3f) ///
   stats(N ll) stfmt(%9.1f) modelwidth(9)

*** ASIDE: Not included in text 

*** Terza two-step estimator
global XLISTEXOG2 age age2 educyr actlim totchr
global y2 private
* Note: need to drop medicaid as probit does not work with Medicaid
* note: medicaid != 0 predicts failure perfectly
*      medicaid dropped and 566 obs not used
probit $y2 $XLISTEXOG2 income ssiratio
predict hat, xb
nl (docvis = exp({xb: private $XLISTEXOG2}+{b0})*( $y2*normal({theta}+hat)/normal(hat) ///
   + (1-$y2)*(1-normal({theta}+hat))/(1-normal(hat)) ))

*********** 10.5.3 TWO ENDOGENOUS REGRESSORS (private and medicaid)

global XLISTEXOG2 age age2 educyr actlim totchr

*** Control function approach

* First stage regression
regress private income ssiratio $XLISTEXOG2, vce(robust)
predict pres1, residuals
regress medicaid income ssiratio $XLISTEXOG2, vce(robust)
predict pres2, residuals

* Poisson with residuals added 
* Standard errors here do not control for 2-step estimation
* Instead use the bootstrap below 
poisson docvis private medicaid $XLISTEXOG2 pres1 pres2, vce(robust)

* Program to return b for Poisson 2-step estimator of selection model 
program poisson2step2, eclass
  version 12
  tempname b V
  capture drop pres1 pres2
  regress private income ssiratio $XLISTEXOG2, vce(robust)
  predict pres1, residuals
  regress medicaid income ssiratio $XLISTEXOG2, vce(robust)
  predict pres2, residuals
  poisson docvis private medicaid $XLISTEXOG2 pres1 pres2
  matrix `b' = e(b)
  ereturn post `b' 
end
* Check preceding program by running once
poisson2step2
ereturn display
* Bootstrap for Poisson two-step estimator
bootstrap _b, reps(400) seed(10101) nowarn: poisson2step2
estimates store PCF2endog

*** GMM

* Additive heterogeneity
gmm ( docvis - exp({xb:private medicaid $XLISTEXOG2}+{b0})),          ///
  instruments(income ssiratio $XLISTEXOG2) onestep vce(robust)
estimates store GMM2add

* Multiplicative heterogeneity
gmm ( (docvis / exp({xb:private medicaid $XLISTEXOG2}+{b0})) - 1),    /// 
  instruments(income ssiratio $XLISTEXOG2) onestep vce(robust)
estimates store GMM2mult

*** Structural model

* Create a combined insurance variable
generate suppins = private + 2*medicaid
label define instype 0 none 1 private 2 medicaid
label values suppins instype
tabulate suppins

* NB1 model 
mtreatreg docvis $XLISTEXOG2, mtreat(suppins = $XLISTEXOG2 income ssiratio) ///
   sim(300) density(negbin1) basecategory(0) vce(robust)
estimates store MMNLNB1

* NB2 model
mtreatreg docvis $XLISTEXOG2, mtreat(suppins = $XLISTEXOG2 income ssiratio) ///
   sim(300) density(negbin2) basecategory(0) vce(robust)
estimates store MMNLNB2

*** RESULTS: TABLE 10.2 TWO ENDOGENOUS REGRESSORS

*** TABLE 10.2 - Poisson CF GMM and MMNL-NB columns
estimates table PCF2endog, b(%7.4f) se(%7.3f) stats(N ll) stfmt(%9.1f) ///
  modelwidth(9) equations(1)

*** TABLE 10.2 (Continued) - GMM columns
estimates table GMM2add GMM2mult, b(%7.4f) se(%7.3f) stats(N ll) stfmt(%9.1f) modelwidth(9)

*** TABLE 10.2 (Continued) - MMNL-NB columns
estimates table MMNLNB1 MMNLNB2, b(%7.4f) se(%7.3f) stats(N ll) stfmt(%9.1f) modelwidth(9)

********** CLOSE OUTPUT

* log close
* clear
* exit
